function devPAV_out = devPAV(lnLR_before_PAV, lnLR_after_PAV)
% calculates devPAV as described in:
% Vergeer P., van Schaik Y., Sjerps M. (2021). Measuring calibration of likelihood-ratio systems: A comparison of four metrics, including a new metric devPAV. Forensic Science International, 321, article 110722. https://doi.org/10.1016/j.forsciint.2021.110722 
% Matlab code is based on R and Python code provided by Vergeer
% 
% Input should be scaled as natural-log likelihood ratios

lnLR_in = [lnLR_before_PAV(:), lnLR_after_PAV(:)];

% remove any ±Inf
II_Inf = lnLR_after_PAV == -Inf | lnLR_after_PAV == Inf;
lnLR_noInf = lnLR_in(~II_Inf,:); 

% unique and sorted
lnLR = unique(lnLR_noInf, 'rows'); 

% log base 10
log10LR = lnLR / log(10);

% x and y values (log10LR_before_PAV and log10LR_after_PAV)
x = log10LR(:,1);
y = log10LR(:,2);

% quit if the input was too short
num_x = length(x);
if num_x < 2
    devPAV_out = NaN;
    return
end

% step through adjacent x values and calculate area
% most of this is a direct copy of the 'calcsurface' function in Vergeer's R and Python code 
%(I copied both R and Python versions row by row to check they were the same, then deleted one)
% it sums the areas of rectangles and triangles
area = zeros(num_x,1);
for I_x = 2:num_x
    
    x1 = x(I_x-1);
    x2 = x(I_x);
    y1 = y(I_x-1);
    y2 = y(I_x);
    
    a = (y2-y1)/(x2-x1);
    if a == 1
        area(I_x) = (y2-y1)*(x2-x1);
    else
        b = y1 - a * x1;
        xs = b / (1-a);
        
        if x1 < xs && x2 >= xs
            
            if y1 <= x1
                area(I_x) = 0.5*(xs-y1)*(xs-x1) - 0.5*(xs-x1)*(xs-x1) + 0.5*(y2-xs)*(x2-xs) - 0.5*(x2-xs)*(x2 - xs);
            else
                area(I_x) = 0.5*(xs-x1)^2 - 0.5*(xs-y1)*(xs-x1) + 0.5*(x2-xs)^2 - 0.5*(x2-xs)*(y2-xs);
            end
            
        else
            
            if y1 <= x1 && y2 <= x1
                area(I_x) = 0.5*(y2-y1)*(x2-x1) + (x1-y2)*(x2-x1) + 0.5*(x2-x1)*(x2-x1);
            elseif y1 > x1
                area(I_x) = 0.5*(x2-x1)*(x2-x1) + (y1-x2)*(x2-x1) + 0.5*(y2-y1)*(x2-x1);
            elseif y1 <= x1 && y2 > x1
                area(I_x) = 0.5*(y2-y1)*(x2-x1) - 0.5*(y2-x1)*(y2-x1) + 0.5*(x2-y2)*(x2-y2);
            else
                error('Error in devPAV')
            end
            
        end
        
    end
        
end

total_area = sum(area);

% full length of coverage, 'a' in Vergeer et al (2021) Fig 2
full_length = x(end) - x(1);

devPAV_out = total_area / full_length;
